<?php
// raporlar/proje_raporu.php — Proje Raporu
// Kaynak DB:
// - proje_fatura_kayitlari (kdv_haric_tutar, genel_toplam, fatura_tipi, tarih)
// - proje_maas_kayitlari (brut_maas, net_maas, sgk_primi, proje_id, yil, ay, personel_id)
declare(strict_types=1);

require_once __DIR__ . '/../auth/guard.php';
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../inc/permissions.php';

@ini_set('default_charset','UTF-8');
mb_internal_encoding('UTF-8');

$uid = (int)($_SESSION['id'] ?? 0);

// raporlar modülü yetkisi
if (!has_module($pdo, $uid, $isAdmin, 'raporlar')) {
  header("Location: " . url('/panel/index.php'));
  exit;
}

if (!function_exists('h')) {
  function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}
function tr_money($n): string { return number_format((float)$n, 2, ',', '.'); }

/**
 * Gelir Vergisi (senin verdiğin dilimler)
 * 0-190.000: %15
 * 190.000-400.000: 28.500 + fazlası %20
 * 400.000-1.500.000: 70.500 + fazlası %27
 * 1.500.000-5.300.000: 367.500 + fazlası %35
 * 5.300.000+: 1.697.500 + fazlası %40
 */
function gelir_vergisi_hesapla(float $matrah): float {
  $m = max(0.0, $matrah);

  if ($m <= 190000)   return $m * 0.15;
  if ($m <= 400000)   return 28500  + ($m - 190000)  * 0.20;
  if ($m <= 1500000)  return 70500  + ($m - 400000)  * 0.27;
  if ($m <= 5300000)  return 367500 + ($m - 1500000) * 0.35;
  return 1697500 + ($m - 5300000) * 0.40;
}

$err = '';

$projeler = $pdo->query("SELECT id, yil, proje_adi FROM projeler ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);

$proje_id = (int)($_GET['proje_id'] ?? 0);
$yil      = (int)($_GET['yil'] ?? (int)date('Y'));
if ($yil < 2000 || $yil > 2100) $yil = (int)date('Y');

$projeAdi = '';
if ($proje_id > 0) {
  foreach ($projeler as $p) {
    if ((int)$p['id'] === $proje_id) { $projeAdi = (string)$p['proje_adi']; break; }
  }
}

$report = [
  'satis' => ['kdv_dahil'=>0.0, 'kdv_haric'=>0.0, 'kdv_toplam'=>0.0],
  'alis'  => ['kdv_dahil'=>0.0, 'kdv_haric'=>0.0, 'kdv_toplam'=>0.0],
  'kdv_fark' => 0.0,

  // Ciro
  'ciro_kdv_haric' => 0.0,

  // Maaş
  'maas_brut_toplam' => 0.0,
  'maas_net_toplam'  => 0.0,
  'maas_sgk_toplam'  => 0.0,

  // Gelir Vergisi (matrah bazlı: Ciro - Brüt Maaş)
  'gv_matrah' => 0.0,
  'gelir_vergisi' => 0.0,
  'ciro_gv_sonrasi' => 0.0,
];

if ($proje_id > 0) {
  try {
    // ==========================
    // 1) FATURA TOPLAMLARI
    // KDV toplamını kolon yok diye hesaplıyoruz: genel_toplam - kdv_haric_tutar
    // ==========================
    $sql = "
      SELECT
        fatura_tipi AS tip,
        COALESCE(SUM(genel_toplam),0) AS kdv_dahil,
        COALESCE(SUM(kdv_haric_tutar),0) AS kdv_haric,
        COALESCE(SUM(genel_toplam - kdv_haric_tutar),0) AS kdv_toplam
      FROM proje_fatura_kayitlari
      WHERE proje_id = :pid
        AND YEAR(tarih) = :yil
      GROUP BY fatura_tipi
    ";
    $st = $pdo->prepare($sql);
    $st->execute([':pid'=>$proje_id, ':yil'=>$yil]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);

    foreach ($rows as $r) {
      $tip = (string)($r['tip'] ?? '');
      $kdv_dahil  = (float)($r['kdv_dahil'] ?? 0);
      $kdv_haric  = (float)($r['kdv_haric'] ?? 0);
      $kdv_toplam = (float)($r['kdv_toplam'] ?? 0);

      if (strcasecmp($tip, 'Satis') === 0) {
        $report['satis']['kdv_dahil']  = $kdv_dahil;
        $report['satis']['kdv_haric']  = $kdv_haric;
        $report['satis']['kdv_toplam'] = $kdv_toplam;
      } elseif (strcasecmp($tip, 'Alis') === 0) {
        $report['alis']['kdv_dahil']  = $kdv_dahil;
        $report['alis']['kdv_haric']  = $kdv_haric;
        $report['alis']['kdv_toplam'] = $kdv_toplam;
      }
    }

    $report['kdv_fark'] = (float)$report['satis']['kdv_toplam'] - (float)$report['alis']['kdv_toplam'];

    // Ciro (KDV hariç) = Satış faturaları KDV hariç toplam
    $report['ciro_kdv_haric'] = (float)$report['satis']['kdv_haric'];

    // ==========================
    // 2) MAAŞ TOPLAMLARI (tüm personel + tüm aylar)
    // ==========================
    $sqlM = "
      SELECT
        COALESCE(SUM(brut_maas),0) AS brut_toplam,
        COALESCE(SUM(net_maas),0)  AS net_toplam,
        COALESCE(SUM(sgk_primi),0) AS sgk_toplam
      FROM proje_maas_kayitlari
      WHERE proje_id = :pid
        AND yil = :yil
    ";
    $stm = $pdo->prepare($sqlM);
    $stm->execute([':pid'=>$proje_id, ':yil'=>$yil]);
    $m = $stm->fetch(PDO::FETCH_ASSOC) ?: [];

    $report['maas_brut_toplam'] = (float)($m['brut_toplam'] ?? 0);
    $report['maas_net_toplam']  = (float)($m['net_toplam'] ?? 0);
    $report['maas_sgk_toplam']  = (float)($m['sgk_toplam'] ?? 0);

    // ==========================
    // 3) GELİR VERGİSİ (Ciro KDV Hariç - Brüt Maaş Toplam)
    // ==========================
    $report['gv_matrah'] = max(0.0, (float)$report['ciro_kdv_haric'] - (float)$report['maas_brut_toplam']);
    $report['gelir_vergisi'] = gelir_vergisi_hesapla((float)$report['gv_matrah']);
    $report['ciro_gv_sonrasi'] = max(0.0, (float)$report['gv_matrah'] - (float)$report['gelir_vergisi']);

  } catch (Throwable $e) {
    $err = "Rapor sorgusu çalışmadı: " . h($e->getMessage());
  }
}

$pageTitle = 'Proje Raporu';
include __DIR__ . '/../inc/header.php';
include __DIR__ . '/../inc/navbar.php';
?>

<div class="container py-4">
  <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mb-3">
    <div>
      <h4 class="fw-bold mb-0"><i class="bi bi-clipboard-data me-1"></i> Proje Raporu</h4>
      <div class="text-secondary small">
        Satış/Alış + KDV + Maaş Toplamları + (Ciro - Brüt Maaş) Gelir Vergisi
      </div>
      <div class="text-secondary small">
        <span class="badge bg-light text-dark border">Fatura: proje_fatura_kayitlari</span>
        <span class="badge bg-light text-dark border ms-1">Maaş: proje_maas_kayitlari</span>
      </div>
    </div>
    <a class="btn btn-outline-secondary" href="<?= h(url('/raporlar/index.php')) ?>">
      <i class="bi bi-arrow-left me-1"></i> Raporlara Dön
    </a>
  </div>

  <?php if ($err): ?>
    <div class="alert alert-danger"><?= $err ?></div>
  <?php endif; ?>

  <div class="card shadow-sm border-0 mb-3">
    <div class="card-body">
      <form method="get" class="row g-3">
        <div class="col-12 col-lg-8">
          <label class="form-label">Proje</label>
          <select class="form-select" name="proje_id" required>
            <option value="">Seç...</option>
            <?php foreach ($projeler as $p): ?>
              <option value="<?= (int)$p['id'] ?>" <?= ($proje_id==(int)$p['id']?'selected':'') ?>>
                #<?= (int)$p['id'] ?> - <?= h($p['proje_adi']) ?> (<?= (int)$p['yil'] ?>)
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-6 col-lg-2">
          <label class="form-label">Yıl</label>
          <input class="form-control" type="number" name="yil" min="2000" max="2100" value="<?= (int)$yil ?>">
        </div>

        <div class="col-6 col-lg-2 d-grid">
          <label class="form-label d-none d-lg-block">&nbsp;</label>
          <button class="btn btn-primary"><i class="bi bi-search me-1"></i> Getir</button>
        </div>
      </form>
    </div>
  </div>

  <?php if ($proje_id > 0 && !$err): ?>
    <div class="alert alert-info d-flex justify-content-between flex-wrap gap-2">
      <div><b>Proje:</b> <?= h($projeAdi ?: ('#'.$proje_id)) ?></div>
      <div><b>Yıl:</b> <?= (int)$yil ?></div>
    </div>

    <div class="row g-3">
      <!-- SATIŞ -->
      <div class="col-12 col-lg-6">
        <div class="card shadow-sm border-0 h-100">
          <div class="card-header bg-white fw-semibold">
            <i class="bi bi-graph-up-arrow me-1 text-success"></i> Satış Faturaları
          </div>
          <div class="card-body">
            <div class="d-flex justify-content-between py-2 border-bottom"><span>KDV Dahil Toplam</span><b><?= tr_money($report['satis']['kdv_dahil']) ?> ₺</b></div>
            <div class="d-flex justify-content-between py-2 border-bottom"><span>KDV Hariç Toplam</span><b><?= tr_money($report['satis']['kdv_haric']) ?> ₺</b></div>
            <div class="d-flex justify-content-between py-2"><span>KDV Toplam</span><b><?= tr_money($report['satis']['kdv_toplam']) ?> ₺</b></div>
          </div>
        </div>
      </div>

      <!-- ALIŞ -->
      <div class="col-12 col-lg-6">
        <div class="card shadow-sm border-0 h-100">
          <div class="card-header bg-white fw-semibold">
            <i class="bi bi-bag-check me-1 text-primary"></i> Alış Faturaları
          </div>
          <div class="card-body">
            <div class="d-flex justify-content-between py-2 border-bottom"><span>KDV Dahil Toplam</span><b><?= tr_money($report['alis']['kdv_dahil']) ?> ₺</b></div>
            <div class="d-flex justify-content-between py-2 border-bottom"><span>KDV Hariç Toplam</span><b><?= tr_money($report['alis']['kdv_haric']) ?> ₺</b></div>
            <div class="d-flex justify-content-between py-2"><span>KDV Toplam</span><b><?= tr_money($report['alis']['kdv_toplam']) ?> ₺</b></div>
          </div>
        </div>
      </div>

      <!-- KDV FARK -->
      <div class="col-12">
        <?php
          $fark = (float)$report['kdv_fark'];
          $cls = ($fark >= 0) ? 'alert-warning' : 'alert-success';
          $label = ($fark >= 0) ? 'Ödenecek KDV (Satış KDV - Alış KDV)' : 'Devreden/İndirilecek KDV (Alış KDV daha fazla)';
        ?>
        <div class="alert <?= $cls ?> mb-0 d-flex justify-content-between align-items-center flex-wrap gap-2">
          <div><i class="bi bi-calculator me-1"></i><b>KDV Farkı:</b> <?= h($label) ?></div>
          <div class="fs-5 fw-bold"><?= tr_money($fark) ?> ₺</div>
        </div>
      </div>

      <!-- MAAŞ TOPLAMLARI -->
      <div class="col-12">
        <div class="card shadow-sm border-0">
          <div class="card-header bg-white fw-semibold">
            <i class="bi bi-people me-1"></i> Personel Toplamları (Tüm Personel + Tüm Aylar)
          </div>
          <div class="card-body">
            <div class="row g-3">
              <div class="col-12 col-lg-4">
                <div class="p-3 rounded border bg-light h-100">
                  <div class="text-secondary small">Toplam Brüt Maaş</div>
                  <div class="fs-4 fw-bold"><?= tr_money($report['maas_brut_toplam']) ?> ₺</div>
                </div>
              </div>
              <div class="col-12 col-lg-4">
                <div class="p-3 rounded border bg-light h-100">
                  <div class="text-secondary small">Toplam Net Ödenen</div>
                  <div class="fs-4 fw-bold"><?= tr_money($report['maas_net_toplam']) ?> ₺</div>
                </div>
              </div>
              <div class="col-12 col-lg-4">
                <div class="p-3 rounded border bg-light h-100">
                  <div class="text-secondary small">Toplam SGK (Tüm Personel)</div>
                  <div class="fs-4 fw-bold"><?= tr_money($report['maas_sgk_toplam']) ?> ₺</div>
                </div>
              </div>
            </div>
            <div class="text-secondary small mt-2">
              Kaynak: <code>proje_maas_kayitlari</code> (proje_id + yil filtreli, ay/personel ayrımı olmadan toplu)
            </div>
          </div>
        </div>
      </div>

      <!-- GELİR VERGİSİ (Ciro - Brüt Maaş) -->
      <div class="col-12">
        <div class="card shadow-sm border-0">
          <div class="card-header bg-white fw-semibold">
            <i class="bi bi-cash-coin me-1"></i> Gelir Vergisi (KDV Hariç Ciro - Brüt Maaş)
          </div>
          <div class="card-body">
            <div class="row g-3">
              <div class="col-12 col-lg-4">
                <div class="p-3 rounded border bg-light h-100">
                  <div class="text-secondary small">Ciro (Satış KDV Hariç)</div>
                  <div class="fs-4 fw-bold"><?= tr_money($report['ciro_kdv_haric']) ?> ₺</div>
                </div>
              </div>
              <div class="col-12 col-lg-4">
                <div class="p-3 rounded border bg-light h-100">
                  <div class="text-secondary small">GV Matrahı (Ciro - Brüt Maaş)</div>
                  <div class="fs-4 fw-bold"><?= tr_money($report['gv_matrah']) ?> ₺</div>
                </div>
              </div>
              <div class="col-12 col-lg-4">
                <div class="p-3 rounded border bg-light h-100">
                  <div class="text-secondary small">Gelir Vergisi (Dilimlere göre)</div>
                  <div class="fs-4 fw-bold text-danger"><?= tr_money($report['gelir_vergisi']) ?> ₺</div>
                </div>
              </div>
              <div class="col-12">
                <div class="alert alert-secondary mb-0 d-flex justify-content-between align-items-center flex-wrap gap-2">
                  <div><b>Matrah - Gelir Vergisi (Kalan)</b></div>
                  <div class="fs-5 fw-bold text-success"><?= tr_money($report['ciro_gv_sonrasi']) ?> ₺</div>
                </div>
              </div>
            </div>

            <div class="text-secondary small mt-2">
              Not: Bu hesap, senin isteğine göre <b>KDV hariç satış cirosundan</b> toplam <b>brüt maaşı</b> düşer ve kalan matraha gelir vergisi dilimi uygular.
            </div>
          </div>
        </div>
      </div>

    </div>
  <?php endif; ?>
</div>

<?php include __DIR__ . '/../inc/footer.php'; ?>
